<?php

declare(strict_types=1);

namespace Gls\GlsPoland\Form\Type;

use Gls\GlsPoland\Configuration\Api;
use Gls\GlsPoland\Translation\TranslatorAwareTrait;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\Extension\Core\Type\HiddenType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\Form\FormEvent;
use Symfony\Component\Form\FormEvents;
use Symfony\Component\Form\FormInterface;
use Symfony\Component\OptionsResolver\OptionsResolver;

final class ApiConfigurationType extends AbstractType
{
    use TranslatorAwareTrait;

    public function buildForm(FormBuilderInterface $builder, array $options): void
    {
        $builder
            ->add('sandboxEnabled', ChoiceType::class, [
                'required' => true,
                'label' => $this->getTranslator()->trans('Operation mode', [], 'Modules.Glspoland.Configuration'),
                'choices' => [
                    $this->getTranslator()->trans('Production', [], 'Modules.Glspoland.Configuration') => false,
                    $this->getTranslator()->trans('Sandbox', [], 'Modules.Glspoland.Configuration') => true,
                ],
                'attr' => [
                    'class' => 'js-sandbox-enabled-choice',
                ],
            ])
            ->add('updateProd', HiddenType::class, [
                'mapped' => false,
            ])
            ->add('updateTest', HiddenType::class, [
                'mapped' => false,
            ])
            ->addEventListener(FormEvents::PRE_SET_DATA, function (FormEvent $event) {
                $form = $event->getForm();

                $this->addProdCredentialsField($form, true);
                $this->addTestCredentialsField($form, true);
            });

        $builder
            ->get('updateProd')
            ->addEventListener(FormEvents::POST_SUBMIT, function (FormEvent $event) {
                $this->addProdCredentialsField($event->getForm()->getParent(), !$event->getData());
            });

        $builder
            ->get('updateTest')
            ->addEventListener(FormEvents::POST_SUBMIT, function (FormEvent $event) {
                $this->addTestCredentialsField($event->getForm()->getParent(), !$event->getData());
            });
    }

    public function configureOptions(OptionsResolver $resolver): void
    {
        $resolver->setDefault('data_class', Api\InMemoryConfiguration::class);
    }

    private function addProdCredentialsField(FormInterface $form, bool $disabled): void
    {
        $form->add('prodCredentials', UserCredentialsType::class, [
            'required' => false,
            'label' => $this->getTranslator()->trans('Production configuration', [], 'Modules.Glspoland.Configuration'),
            'disabled' => $disabled,
            'attr' => [
                'class' => 'js-user-credentials-form',
                'data-toggle-field-name' => 'updateProd',
                'data-sandbox' => '0',
            ],
        ]);
    }

    private function addTestCredentialsField(FormInterface $form, bool $disabled): void
    {
        $form->add('testCredentials', UserCredentialsType::class, [
            'required' => false,
            'label' => $this->getTranslator()->trans('Sandbox configuration', [], 'Modules.Glspoland.Configuration'),
            'disabled' => $disabled,
            'attr' => [
                'class' => 'js-user-credentials-form',
                'data-toggle-field-name' => 'updateTest',
                'data-sandbox' => '1',
            ],
        ]);
    }
}
