<?php

declare(strict_types=1);

namespace Gls\GlsPoland\PrestaShop\DTO;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;

class Carrier
{
    private $carrier;
    private $groups;
    private $zones;
    private $ranges;
    private $disallowedPaymentModuleIds = [];
    private $allowedPaymentModuleIds;
    private $taxRulesGroup;

    public function __construct(\Carrier $carrier)
    {
        $this->carrier = $carrier;
        $this->groups = new ArrayCollection();
        $this->zones = new ArrayCollection();
        $this->ranges = new ArrayCollection();
    }

    public function getCarrier(): \Carrier
    {
        return $this->carrier;
    }

    /**
     * @return Collection<\Group>
     */
    public function getGroups(): Collection
    {
        return $this->groups;
    }

    public function addGroup(\Group $group): self
    {
        if (!$this->groups->contains($group)) {
            $this->groups->add($group);
        }

        return $this;
    }

    public function removeGroup(\Group $group): self
    {
        if ($this->groups->contains($group)) {
            $this->groups->removeElement($group);
        }

        return $this;
    }

    /**
     * @return Collection<\Zone>
     */
    public function getZones(): Collection
    {
        return $this->zones;
    }

    public function addZone(\Zone $zone): self
    {
        if (!$this->zones->contains($zone)) {
            $this->zones->add($zone);
        }

        return $this;
    }

    public function removeZone(\Zone $zone): self
    {
        if ($this->zones->contains($zone)) {
            $this->zones->removeElement($zone);
        }

        return $this;
    }

    /**
     * @return Collection<\RangeWeight>
     */
    public function getRanges(): Collection
    {
        return $this->ranges;
    }

    public function addRange(\RangeWeight $range): self
    {
        if (!$this->ranges->contains($range)) {
            $this->ranges->add($range);
        }

        return $this;
    }

    public function removeRange(\RangeWeight $range): self
    {
        if ($this->ranges->contains($range)) {
            $this->ranges->removeElement($range);
        }

        return $this;
    }

    /**
     * @return int[]
     */
    public function getDisallowedPaymentModuleIds(): array
    {
        return $this->disallowedPaymentModuleIds;
    }

    public function setDisallowedPaymentModulesIds(array $ids): self
    {
        $this->disallowedPaymentModuleIds = [];
        foreach ($ids as $id) {
            $this->addDisallowedPaymentModuleId($id);
        }

        return $this;
    }

    public function addDisallowedPaymentModuleId(int $id): self
    {
        if (0 < $id) {
            $this->disallowedPaymentModuleIds[$id] = $id;
            unset($this->allowedPaymentModuleIds[$id]);
        }

        return $this;
    }

    public function removeDisallowedPaymentModuleId(int $id): self
    {
        unset($this->disallowedPaymentModuleIds[$id]);

        return $this;
    }

    /**
     * @return int[]|null
     */
    public function getAllowedPaymentModuleIds(): ?array
    {
        return $this->allowedPaymentModuleIds;
    }

    /**
     * @param int[]|null $ids list of IDs or null if all payment modules except explicitly disallowed should be available
     */
    public function setAllowedPaymentModulesIds(?array $ids): self
    {
        if (null === $ids) {
            $this->allowedPaymentModuleIds = null;

            return $this;
        }

        $this->allowedPaymentModuleIds = [];
        foreach ($ids as $id) {
            $this->addAllowedPaymentModuleId($id);
        }

        return $this;
    }

    public function addAllowedPaymentModuleId(int $id): self
    {
        if (0 < $id) {
            $this->allowedPaymentModuleIds[$id] = $id;
            unset($this->disallowedPaymentModuleIds[$id]);
        }

        return $this;
    }

    public function removeAllowedPaymentModuleId(int $id): self
    {
        unset($this->allowedPaymentModuleIds[$id]);

        return $this;
    }

    public function getTaxRulesGroup(): ?\TaxRulesGroup
    {
        return $this->taxRulesGroup;
    }

    public function setTaxRulesGroup(\TaxRulesGroup $taxRulesGroup): self
    {
        $this->taxRulesGroup = $taxRulesGroup;

        return $this;
    }
}
